'use strict';

/* --------------------------------------------------------------
 swiper.js 2024-06-14
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2024 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/* globals Swiper */

/**
 * Widget that binds the swiper plugin (third party) to a DOM element
 *
 * @todo Remove the try - catch blocks and and correct the swiper issues.
 */
gambio.widgets.module('swiper', [gambio.source + '/libs/events', gambio.source + '/libs/responsive'], function (data) {

    'use strict';

    // ########## VARIABLE INITIALIZATION ##########

    var $this = $(this),
        $body = $('body'),
        $slides = null,
        $controls = null,
        $target = null,
        $theme = null,
        init = true,
        swiper = null,
        sliderOptions = null,
        hasThumbnails = true,
        mode = null,
        breakpointDataset = null,
        duplicates = false,
        preventSlideStart = false,
        sliderDefaults = {
        observer: true,
        // Default configuration for the swiper
        pagination: '.swiper-pagination',
        nextButton: '.swiper-button-next',
        prevButton: '.swiper-button-prev',
        paginationClickable: true,
        loop: true,
        autoplay: 3,
        autoplayDisableOnInteraction: false
    },
        defaults = {
        observer: true,
        // JSON that gets merged with the sliderDefaults and is passed to "swiper" directly.
        sliderOptions: null,
        // If this instance is a "main" swiper, the given selector selects the "control" swiper.
        controls: null,
        // If this instance is a "control" swiper, the given selector selects the "main" swiper.
        target: null,
        // Sets the initial slide (needed to prevent different init slides in main/controller slider).
        initSlide: null,
        // Detect if a swiper is needed for the breakpoint. If not, turn it off
        autoOff: false,
        // The translucence fix enables support for a fade effect between images with different aspect ratio,
        // but causing a delay between the change
        disableTranslucenceFix: false,
        breakpoints: []
    },
        options = $.extend({}, defaults, data),
        module = {};

    // ########## HELPER FUNCTIONS ##########

    /**
     * Function that generates the markup for
     * the preview bullets
     * @param       {Swiper}        swiper          Swiper object
     * @param       {integer}       index           Index of the slide
     * @param       {string}        className       The classname that must be add to the markup
     * @return      {string}                        The preview image html string
     * @private
     */
    var _generatePreviewButtons = function _generatePreviewButtons(swiper, index, className) {
        var $currentSlide = $slides.eq(index),
            $image = $currentSlide.find('img'),
            altTxt = $image.attr('alt'),
            thumbImage = $currentSlide.data('thumbImage');

        if (thumbImage) {
            return '<img src="' + thumbImage + '" alt="' + altTxt + '" class="' + className + '" loading="lazy" />';
        }

        return '';
    };

    /**
     * Helper function to get the index of the
     * active slide
     * @return     {integer}                       The index of the active slide
     * @private
     */
    var _getIndex = function _getIndex() {
        return $this.find('.swiper-slide-active').data("swiper-slide-index") || 0;
    };

    /**
     * Helper function to add the active
     * class to the active slide
     * @param       {integer}           index       The index of the active slide
     * @private
     */
    var _setActive = function _setActive(index) {
        $slides = $this.find('.swiper-slide:not(.swiper-slide-duplicate)');
        index = duplicates ? index + 1 : index;
        $slides.removeClass('active').eq(index).addClass('active');
    };

    // ########## EVENT HANDLER ##########

    /**
     * Event handler for the mouseenter event.
     * It disables the autoplay
     * @private
     */
    var _mouseEnterHandler = function _mouseEnterHandler() {
        try {
            if (swiper) {
                swiper.stopAutoplay();
            }
        } catch (e) {
            // Do not log the error
        }
    };

    /**
     * Event handler for the mouseleave event.
     * It enables the autoplay
     * @private
     */
    var _mouseLeaveHandler = function _mouseLeaveHandler() {
        try {
            if (swiper) {
                swiper.startAutoplay();
            }
        } catch (e) {
            // Do not log the error
        }
    };

    /**
     * Event handler for the goto event.
     * It switches the current slide to the given index
     * and adds the active class to the new active slide
     * @param       {object}    e       jQuery event object
     * @param       {number}    d       Index of the slide to show
     * @private
     */
    var _gotoHandler = function _gotoHandler(e, d) {
        e.stopPropagation();

        // Set the active slide
        _setActive(d);

        // Temporary deactivate the onSlideChangeStart event
        // to prevent looping through the goto / changeStart
        // events
        preventSlideStart = true;

        // Remove the autoplay after a goto event
        $this.off('mouseleave.swiper');
        swiper.stopAutoplay();

        // Try to correct the index between sliders
        // with and without duplicates
        var index = duplicates ? d + 1 : d;
        if (index > $slides.length) {
            index = 0;
        }

        // Goto the desired slide
        swiper.slideTo(index);

        // Reactivate the onSlideChangeEvent
        preventSlideStart = false;
    };

    /**
     * Click event handler that triggers a
     * "goto" event to the target swiper
     * @param       {object}        e       jQuery event object
     * @private
     */
    var _clickHandler = function _clickHandler(e) {
        e.preventDefault();
        e.stopPropagation();

        var $self = $(this),
            index = $self.index();

        index = duplicates ? index - 1 : index;

        // Set the active slide
        _setActive(index);

        // Inform the main swiper
        $target.trigger(jse.libs.theme.events.SWIPER_GOTO(), index);
    };

    /**
     * Event that gets triggered on slideChange.
     * If the slide gets changed, the controls
     * will follow the current slide in position
     * @private
     */
    var _triggerSlideChange = function _triggerSlideChange() {
        if (!preventSlideStart) {
            var index = _getIndex(),
                lastIndex = $slides.closest(".swiper-slide-prev").index() || $slides.length - 2;

            var previousSlider = $slides.closest(".swiper-slide-prev");
            if (previousSlider) {
                lastIndex = duplicates ? previousSlider.index() - 1 : previousSlider.index();
            }

            if (index > $slides.length - 1) {
                index = 0;
            }

            // Recalculate index if duplicate slides are inside the slider
            if (index < 0) {
                index = $slides.length + index;
            } else {
                index = duplicates && index === lastIndex ? index - lastIndex : index;
            }

            // Set the active slide
            _setActive(index);

            // Inform the controls
            $controls.trigger(jse.libs.theme.events.SWIPER_GOTO(), index);
        }
    };

    /**
     * Workaround for the translucence issue
     * that happens on small screens with enabled
     * fade effect. Maybe it can be removed, if the
     * swiper gets updated itself
     * @private
     */
    var _translucenceWorkaround = function _translucenceWorkaround() {
        if (!options.disableTranslucenceFix && sliderOptions && sliderOptions.effect === 'fade') {
            $this.find('.swiper-slide').filter(':not(.swiper-slide-active)').fadeTo(300, 0, function () {
                $(this).css('visibility', 'hidden');
            });

            $this.find('.swiper-slide').filter('.swiper-slide-active').fadeTo(300, 1, function () {
                $(this).css('visibility', '');
            });
        }
    };

    /**
     * The breakpoint handler initializes the swiper
     * with the settings for the current breakpoint.
     * Therefore it uses the default slider options,
     * the custom slider options given by the options
     * object and the breakpoint options object also
     * given by the options (in this order)
     * @private
     */
    var _breakpointHandler = function _breakpointHandler() {

        // Get the current viewmode
        var oldMode = mode || {},
            newMode = jse.libs.theme.responsive.breakpoint(),
            extendOptions = options.breakpoints[0] || {},
            newBreakpointDataset = null;

        // Only do something if the view was changed
        if (newMode.id !== oldMode.id) {

            // Store the new viewmode
            mode = $.extend({}, newMode);

            // Iterate through the breakpoints object to detect
            // the correct settings for the current breakpoint
            $.each(options.breakpoints, function (i, v) {
                if (v.breakpoint > newMode.id) {
                    return false;
                }
                newBreakpointDataset = i;
                extendOptions = v;
            });

            if (options.sliderOptions && options.sliderOptions.breakpoints) {
                $.each(options.sliderOptions.breakpoints, function (i, v) {
                    if (v.breakpoint === newMode.id) {
                        extendOptions = v;
                        return false;
                    }
                });
            }

            // Only do something if the settings change due browser
            // resize or if it's the first time run
            if (newBreakpointDataset !== breakpointDataset || init) {
                // Combine the settings
                sliderOptions = $.extend({}, sliderDefaults, options.sliderOptions || {}, extendOptions);

                // Add the preview image bullets function to the options object
                if (sliderOptions.usePreviewBullets && hasThumbnails) {
                    sliderOptions.paginationBulletRender = _generatePreviewButtons;
                }

                // Add the autoplay interval to the options object
                sliderOptions.autoplay = sliderOptions.autoplay ? sliderOptions.autoplay * 1000 : 0;

                // Disable loop if there is only one slider. 
                if ($this.find('.swiper-slide').length === 1) {
                    sliderOptions.loop = false;
                }

                // If an swiper exists, get the current
                // slide no. and remove the old swiper
                if (swiper) {
                    sliderOptions.initialSlide = _getIndex();
                    try {
                        swiper.destroy(true, true);
                    } catch (ignore) {
                        swiper = null;
                    }
                } else {
                    sliderOptions.initialSlide = options.initSlide || sliderOptions.initialSlide || 0;
                }

                var $duplicate = $this.find('.swiper-slide:not(.swiper-slide-duplicate)');

                if (!options.autoOff || $duplicate.length > sliderOptions.slidesPerView && options.autoOff) {
                    $this.addClass('swiper-is-active').removeClass('swiper-is-not-active');

                    // Initialize the swiper
                    try {
                        swiper = new Swiper($this, sliderOptions);
                    } catch (e) {
                        return; // Swiper might throw an error upon initialization that should not halt the script execution.
                    }

                    swiper.off('onTransitionEnd onSlideChangeStart').on('onTransitionEnd', _translucenceWorkaround);

                    // If this is a "main" swiper and has external controls, an
                    // goto event is triggered if the current slide is changed
                    if ($controls.length) {
                        swiper.on('onSlideChangeStart', _triggerSlideChange);
                    }

                    // Add the event handler
                    $this.off('mouseenter.swiper mouseleave.swiper ' + jse.libs.theme.events.SWIPER_GOTO() + ' ' + jse.libs.theme.events.SLIDES_UPDATE()).on('mouseenter.swiper', _mouseEnterHandler).on('mouseleave.swiper', _mouseLeaveHandler).on(jse.libs.theme.events.SWIPER_GOTO(), _gotoHandler).on(jse.libs.theme.events.SLIDES_UPDATE(), _updateSlides);

                    if (init) {
                        // Check if there are duplicates slides (generated by the swiper)
                        // after the first time init of the swiper
                        duplicates = !!$this.find('.swiper-slide-duplicate').length;
                    }

                    // Set the active slide
                    var index = init && options.initSlide ? options.initSlide : _getIndex();
                    _setActive(index);

                    // Inform the controls that the main swiper has changed
                    // In case that the other slider isn't initialized yet,
                    // set an data attribute to the markup element to inform
                    // it on init
                    if ($controls.length) {
                        $controls.attr('data-swiper-init-slide', index);
                        _triggerSlideChange();
                    }

                    // Unset the init flag
                    init = false;

                    setTimeout(function () {
                        return swiper.update();
                    });
                } else {
                    // Disable the swiper buttons
                    $this.removeClass('swiper-is-active').addClass('swiper-is-not-active');
                    init = true;
                }
            }
        }
    };

    /**
     * Event handler that adds & removes slides from the
     * swiper. After the slides were processed, the first
     * slide is shown
     * @param       {object}    e       jQuery event object
     * @param       {object}    d       JSON object that contains the categories / images
     * @private
     */
    var _updateSlides = function _updateSlides(e, d) {

        // Loops through each category inside the images array
        $.each(d, function (category, dataset) {
            var catName = category + '-category',
                add = [],
                remove = [],
                markup = $theme.html();

            // Get all indexes from the slides
            // of the same category and remove
            // them from the slider
            $slides.filter('.' + catName).each(function () {
                var $self = $(this),
                    index = $self.data().swiperSlideIndex;

                index = index === undefined ? $self.index() : index;
                remove.push(index);
            });
            swiper.removeSlide(remove);

            // Generate the markup for the new slides
            // and add them to the slider
            $.each(dataset || [], function (i, v) {
                v.className = catName;
                v.srcattr = 'src="' + v.src + '"';
                add.push(Mustache.render(markup, v));
            });
            swiper.appendSlide(add);
        });

        $slides = $this.find('.swiper-slide');

        // To prevent an inconsistent state
        // in control / main slider combinations
        // slide to the first slide
        _setActive(0);
        var index = duplicates ? 1 : 0;
        swiper.slideTo(index, 0);
    };

    /**
     * Prevent text selection by clicking on swiper buttons
     * @private
     */
    var _preventTextSelection = function _preventTextSelection() {
        $(options.sliderOptions.nextButton).on('selectstart', function () {
            return false;
        });
        $(options.sliderOptions.prevButton).on('selectstart', function () {
            return false;
        });
    };

    /**
     * Sets the initial height for one swiper image container to prevent cut off images on smaller swiper heights
     * @private
     */
    var _scaleThumbnailHeight = function _scaleThumbnailHeight() {
        var swiperContainer = $('.swiper-container-vertical .swiper-slide');
        var $containerHeight = swiperContainer.css('height');

        // Workaround for IE Browsers
        if ($('.swiper-container-vertical').hasClass('swiper-wp8-vertical')) {
            $containerHeight = swiperContainer.height() + 5;

            swiperContainer.css('height', $containerHeight);
        }

        if ($containerHeight === '0px') {
            $containerHeight = $('.product-info-thumbnails-mobile').css('height');
        }

        $('.align-middle').css('height', $containerHeight);
    };

    // ########## INITIALIZATION ##########

    /**
     * Init function of the widget
     * @constructor
     */
    module.init = function (done) {

        $slides = $this.find('.swiper-slide');
        $controls = $(options.controls);
        $target = $(options.target);
        $theme = $this.find('theme');

        // Check if all images inside the swiper have
        // thumbnail image given
        $slides.each(function () {
            if (!$(this).data().thumbImage) {
                hasThumbnails = false;
                return false;
            }
        });

        // Add the breakpoint handler ty dynamically
        // set the options corresponding to the browser size (slider responsive will re-initialize the swiper).
        _breakpointHandler();

        // If this instance is a "control" swiper the target is the main swiper
        // which will be updated on a click inside this control swiper
        if (options.target) {
            $this.on('click.swiper', '.swiper-slide', _clickHandler);
        }

        $(document).ready(function () {
            $('.swiper-vertical .swiper-slide[data-index]').css('display', 'inline-block');
            $('.product-info-image .swiper-slide[data-index]').css('z-index', 'inherit');
            $('.product-info-image .swiper-slide[data-index] .swiper-slide-inside img.img-responsive').fadeIn(1000);
        });

        _translucenceWorkaround();
        _preventTextSelection();
        _scaleThumbnailHeight();

        // Fix for invisible Thumbnail-Images for switching from Tablet-Portrait to Tablet-Landscape
        $body.on(jse.libs.theme.events.BREAKPOINT(), function () {
            _scaleThumbnailHeight();
        });

        done();
    };

    // Return data to widget engine
    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
